/*
 * Decompiled with CFR 0.152.
 */
package org.systemsbiology.chem;

import cern.jet.random.Gamma;
import java.util.HashMap;
import java.util.LinkedList;
import org.systemsbiology.chem.Reaction;
import org.systemsbiology.chem.Species;
import org.systemsbiology.chem.SymbolEvaluatorChem;
import org.systemsbiology.data.AbstractComparator;
import org.systemsbiology.data.ListQueue;
import org.systemsbiology.data.PriorityQueue;
import org.systemsbiology.data.Queue;
import org.systemsbiology.data.SlidingWindowTimeSeriesQueue;
import org.systemsbiology.math.MutableDouble;
import org.systemsbiology.math.Symbol;
import org.systemsbiology.math.SymbolEvaluator;
import org.systemsbiology.math.SymbolValue;
import org.systemsbiology.util.DataNotFoundException;

public final class DelayedReactionSolver {
    private static final double LAMBDA_MAX = 1.1;
    public static final int MIN_NUM_HISTORY_BINS = 10;
    public static final int DEFAULT_NUM_HISTORY_BINS = 400;
    private final Species mReactant;
    private final Species mIntermedSpecies;
    private final double mRate;
    private double mTimeResolution;
    private boolean mFirstTimePoint;
    private boolean mIsMultistep;
    private final Queue mReactionTimes;
    private final LinkedList mReactionTimesDoublePool;
    private SlidingWindowTimeSeriesQueue mReactantHistory;
    private int mNumTimePoints;
    private final boolean mIsStochasticSimulator;
    private final int mReactionIndex;
    private final double mDelay;

    public String toString() {
        return this.mIntermedSpecies.getName();
    }

    public double getRate() {
        return this.mRate;
    }

    public double getDelay() {
        return this.mDelay;
    }

    public Species getIntermedSpecies() {
        return this.mIntermedSpecies;
    }

    public int getNumHistoryBins() {
        return this.mNumTimePoints;
    }

    public void setNumHistoryBins(int pNumHistoryBins) {
        if (pNumHistoryBins < 10) {
            throw new IllegalArgumentException("invalid history bin size, must be at least 10");
        }
        this.mNumTimePoints = pNumHistoryBins;
        this.mTimeResolution = 1.1 * this.mDelay / (double)pNumHistoryBins;
        if (this.mIsStochasticSimulator) {
            this.mReactionTimesDoublePool.clear();
            int ctr = 0;
            while (ctr < pNumHistoryBins) {
                this.mReactionTimesDoublePool.add(new MutableDouble(0.0));
                ++ctr;
            }
        } else {
            this.mReactantHistory.initialize(this.mNumTimePoints);
        }
    }

    public DelayedReactionSolver(Species pReactant, Species pIntermedSpecies, double pDelay, double pRate, boolean pIsMultistep, int pReactionIndex, boolean pIsStochasticSimulator) {
        assert (pDelay > 0.0) : "invalid delay";
        this.mDelay = pDelay;
        this.mReactant = pReactant;
        this.mIntermedSpecies = pIntermedSpecies;
        this.mRate = pRate;
        this.mFirstTimePoint = true;
        this.mIsMultistep = pIsMultistep;
        this.mReactionIndex = pReactionIndex;
        this.mIsStochasticSimulator = pIsStochasticSimulator;
        if (this.mIsStochasticSimulator) {
            this.mReactionTimes = this.mIsMultistep ? new PriorityQueue(new AbstractComparator(){

                public final int compare(Object p1, Object p2) {
                    return MutableDouble.compare((MutableDouble)p1, (MutableDouble)p2);
                }
            }) : new ListQueue();
            this.mReactionTimesDoublePool = new LinkedList();
            this.mReactantHistory = null;
        } else {
            this.mReactantHistory = new SlidingWindowTimeSeriesQueue(1);
            this.mReactionTimes = null;
            this.mReactionTimesDoublePool = null;
        }
        this.setNumHistoryBins(400);
    }

    int getReactionIndex() {
        return this.mReactionIndex;
    }

    void addReactant(SymbolEvaluatorChem pSymbolEvaluator) {
        MutableDouble newReactionTime = null;
        double relTime = this.mIsMultistep ? Gamma.staticNextDouble((double)(this.mRate * this.mDelay), (double)this.mRate) : this.mDelay;
        double reactionTime = pSymbolEvaluator.getTime() + relTime;
        LinkedList reactionTimesDoublePool = this.mReactionTimesDoublePool;
        if (reactionTimesDoublePool.size() > 0) {
            newReactionTime = (MutableDouble)reactionTimesDoublePool.getLast();
            reactionTimesDoublePool.removeLast();
            newReactionTime.setValue(reactionTime);
        } else {
            newReactionTime = new MutableDouble(reactionTime);
        }
        this.mReactionTimes.add(newReactionTime);
    }

    double pollNextReactionTime() throws IllegalStateException {
        MutableDouble reactionTime = (MutableDouble)this.mReactionTimes.getNext();
        if (reactionTime == null) {
            throw new IllegalStateException("no molecules are in the multistep reaction queue");
        }
        double nextReactionTime = reactionTime.getValue();
        reactionTime.setValue(0.0);
        this.mReactionTimesDoublePool.addLast(reactionTime);
        return nextReactionTime;
    }

    boolean canHaveReaction() {
        return this.mReactionTimes.peekNext() != null;
    }

    double peekNextReactionTime() throws IllegalStateException {
        MutableDouble reactionTime = (MutableDouble)this.mReactionTimes.peekNext();
        if (reactionTime == null) {
            throw new IllegalStateException("no molecules are in the multistep reaction queue");
        }
        return reactionTime.getValue();
    }

    double getEstimatedAverageFutureRate(SymbolEvaluator pSymbolEvaluator) throws DataNotFoundException {
        double rate = pSymbolEvaluator.getValue(this.mIntermedSpecies.getSymbol()) / this.mDelay;
        return rate;
    }

    void clear() {
        if (this.mIsStochasticSimulator) {
            while (this.mReactionTimes.peekNext() != null) {
                MutableDouble reactionTime = (MutableDouble)this.mReactionTimes.getNext();
                reactionTime.setValue(0.0);
                this.mReactionTimesDoublePool.addLast(reactionTime);
            }
        } else {
            this.mReactantHistory.clear();
            this.mFirstTimePoint = true;
        }
    }

    void initializeSpeciesSymbols(HashMap pSymbolMap, Species[] pDynamicSymbolValues, SymbolValue[] pNonDynamicSymbolValues) throws IllegalStateException {
        Symbol intermedSymbol = this.mIntermedSpecies.getSymbol();
        intermedSymbol.copyIndexInfo(Reaction.getIndexedSpecies(this.mIntermedSpecies, pSymbolMap, pDynamicSymbolValues, pNonDynamicSymbolValues).getSymbol());
        Symbol reactantSymbol = this.mReactant.getSymbol();
        reactantSymbol.copyIndexInfo(Reaction.getIndexedSpecies(this.mReactant, pSymbolMap, pDynamicSymbolValues, pNonDynamicSymbolValues).getSymbol());
    }

    public void update(SymbolEvaluator pSymbolEvaluator, double pTime) throws DataNotFoundException {
        SlidingWindowTimeSeriesQueue reactantHistory = this.mReactantHistory;
        if (!this.mFirstTimePoint) {
            double lastTime = reactantHistory.getLastTimePoint();
            boolean gotValue = false;
            double reactantValue = 0.0;
            double intermedSpeciesValue = 0.0;
            while (pTime - lastTime > this.mTimeResolution) {
                if (!gotValue) {
                    reactantValue = pSymbolEvaluator.getValue(this.mReactant.getSymbol());
                    intermedSpeciesValue = pSymbolEvaluator.getValue(this.mIntermedSpecies.getSymbol());
                }
                lastTime += this.mTimeResolution;
                assert (reactantValue >= 0.0) : "invalid value";
                reactantHistory.insertPoint(lastTime, reactantValue);
                assert (intermedSpeciesValue >= 0.0) : "invalid value";
            }
        } else {
            double reactantValue = pSymbolEvaluator.getValue(this.mReactant.getSymbol());
            assert (reactantValue >= 0.0) : "invalid value";
            double intermedSpeciesValue = pSymbolEvaluator.getValue(this.mIntermedSpecies.getSymbol());
            assert (intermedSpeciesValue >= 0.0) : "invalid value";
            reactantHistory.insertPoint(pTime, reactantValue);
            this.mFirstTimePoint = false;
        }
    }

    public double computeRate(SymbolEvaluator pSymbolEvaluator) throws DataNotFoundException {
        if (!this.mIsStochasticSimulator) {
            if (this.mIsMultistep) {
                return this.computeRateMultistep(pSymbolEvaluator);
            }
            return this.computeRateDelay(pSymbolEvaluator);
        }
        return 0.0;
    }

    private double computeRateMultistep(SymbolEvaluator pSymbolEvaluator) throws DataNotFoundException {
        double prodRate = 0.0;
        SymbolEvaluatorChem symbolEvaluator = (SymbolEvaluatorChem)pSymbolEvaluator;
        double currentTime = symbolEvaluator.getTime();
        SlidingWindowTimeSeriesQueue reactantSpeciesHistory = this.mReactantHistory;
        double intermedSpeciesValue = symbolEvaluator.getValue(this.mIntermedSpecies.getSymbol());
        if (intermedSpeciesValue > 0.0) {
            prodRate = DelayedReactionSolver.computeIntegral(reactantSpeciesHistory, this.mTimeResolution, this.mNumTimePoints, this.mDelay, this.mRate, currentTime);
        }
        return prodRate;
    }

    private double computeRateDelay(SymbolEvaluator pSymbolEvaluator) throws DataNotFoundException {
        double prodRate = 0.0;
        SymbolEvaluatorChem symbolEvaluator = (SymbolEvaluatorChem)pSymbolEvaluator;
        double currentTime = symbolEvaluator.getTime();
        SlidingWindowTimeSeriesQueue reactantSpeciesHistory = this.mReactantHistory;
        double intermedSpeciesValue = symbolEvaluator.getValue(this.mIntermedSpecies.getSymbol());
        double minTime = reactantSpeciesHistory.getMinTime();
        double peakTimeRel = this.mDelay;
        double peakTime = currentTime - peakTimeRel;
        if (intermedSpeciesValue > 0.0 && peakTime >= minTime) {
            double peakValue = 0.0;
            double peakIndexDouble = (peakTime - minTime) / this.mTimeResolution;
            double peakIndexDoubleFloor = Math.floor(peakIndexDouble);
            int peakIndex = (int)peakIndexDouble;
            if (peakIndexDouble > peakIndexDoubleFloor) {
                double valueLeft = reactantSpeciesHistory.getValue(peakIndex);
                peakValue = valueLeft + (peakIndexDouble - peakIndexDoubleFloor) * (reactantSpeciesHistory.getValue(peakIndex + 1) - valueLeft);
            } else {
                peakValue = reactantSpeciesHistory.getValue(peakIndex);
            }
            prodRate = this.mRate * peakValue;
        }
        return prodRate;
    }

    private static double computeIntegral(SlidingWindowTimeSeriesQueue history, double timeResolution, int numTimePoints, double delay, double rate, double currentTime) {
        double value = 0.0;
        double prodRate = 0.0;
        double numStepsCorrected = delay * rate;
        int numPoints = history.getNumStoredPoints();
        double sqrtTwoPiNumStepsCorrected = Math.sqrt(Math.PI * 2 * numStepsCorrected);
        int ctr = numPoints;
        while (--ctr >= 0) {
            value = timeResolution * DelayedReactionSolver.computeIntegrandValue(history, ctr, rate, rate * rate, sqrtTwoPiNumStepsCorrected, numStepsCorrected, currentTime);
            if (ctr == 0 || ctr == numTimePoints - 1) {
                prodRate += value / 3.0;
                continue;
            }
            if (ctr % 2 == 1) {
                prodRate += 2.0 * value / 3.0;
                continue;
            }
            prodRate += 4.0 * value / 3.0;
        }
        return prodRate;
    }

    private static double computeIntegrandValue(SlidingWindowTimeSeriesQueue pReactantHistory, int pTimePointIndex, double pRate, double pRateSquared, double pSqrtTwoPiNumStepsCorrected, double numStepsCorrected, double pCurrentTime) {
        double reactantValue = pReactantHistory.getValue(pTimePointIndex);
        double timePoint = pReactantHistory.getTimePoint(pTimePointIndex);
        assert (pCurrentTime >= timePoint) : "time point is in the future";
        double rate = pRate;
        double lambda = rate * (pCurrentTime - timePoint);
        double retVal = reactantValue * pRateSquared * Math.pow(lambda * Math.E / numStepsCorrected, numStepsCorrected) / (Math.exp(lambda) * pSqrtTwoPiNumStepsCorrected);
        return retVal;
    }
}

